#include "stdafx.h"
#include "version.h"

#include "rwexp.h"
#include "utilities.h"
#include "material.h"

RpMaterial *Max2RpMaterial(Mtl *maxMaterial, int matID, RwInt32 textureNameCase);
void GetFileName(char *outFileName, char *fullPathName, RwInt32 nameCase);

RpMaterial *
CompareRpMaterial2MaxMaterial(RpMaterial *rpMaterial, void *pData)
{    
    matSearch *info = (matSearch *)pData;
    BitmapTex *maxTexture = NULL;
    BitmapTex *maskTexture = NULL;
    RwTexture *rwTexture = NULL;
    Texmap *texmap = NULL;
    Mtl *maxMaterial = GetSubMaterial(info->maxMaterial, info->maxMatID);

    if (!maxMaterial)
    {
        /* compare to white */
        if (!RpMaterialGetTexture(rpMaterial) &&
            rpMaterial->color.red == 255 &&
            rpMaterial->color.green == 255 &&
            rpMaterial->color.blue == 255 &&
            rpMaterial->color.alpha == 255)
        {
            info->material = rpMaterial;
            return(NULL);
        }
        else
        {
            return (rpMaterial);
        }
    }

    /* get any RW texture */
    rwTexture = RpMaterialGetTexture(rpMaterial);
    /* get any max texture */
    
    texmap = GetStdMatTexmap(maxMaterial, ID_DI);

    if (texmap && texmap->ClassID() == Class_ID(BMTEX_CLASS_ID, 0))
    {
        maxTexture = (BitmapTex *)texmap;
    }
     
    /* are both textured */
    if (maxTexture && rwTexture)
    {
        /* compare textures */
        {
            char *fullTextureName;
            char maxTextureName[256];
            char *rwTextureName;

            /* get the max texture name */
            fullTextureName = maxTexture->GetMapName();
            GetFileName(maxTextureName, fullTextureName, info->nameCase);

            if (strlen(maxTextureName) > (rwTEXTUREBASENAMELENGTH-1))
            {
                maxTextureName[rwTEXTUREBASENAMELENGTH-1] = 0;
            }            

            /* get the rw texture name */
            rwTextureName = RwTextureGetName(rwTexture);

            /* do both have the same texture */
            if (stricmp(maxTextureName, rwTextureName) != 0)
            {
                /* not equal */
                return (rpMaterial);
            }
        }
    }    
    /* else if neither are textured or both are textured and have
       compared equal compare colour and opacity */
    if ((!maxTexture && !rwTexture) || (maxTexture && rwTexture))
    {
        /* compare opacity and colour */
        {
            RwRGBA cColour;
            const RwRGBA *rwColour;

            /* get max opacity & colour*/
            MaterialGetColour(maxMaterial, &cColour, FALSE);

            rwColour = RpMaterialGetColor(rpMaterial);

            if ((rwColour->alpha != cColour.alpha) ||
                (!maxTexture &&
                 ((rwColour->red != cColour.red) ||
                  (rwColour->green != cColour.green) ||
                  (rwColour->blue != cColour.blue))))
            {
                return (rpMaterial);
            }            
        }

        info->material = rpMaterial;
        return (NULL);
    }

    return (rpMaterial);
}

void
CheckMaterialMask(RpMaterial *material, Mtl *maxMaterial, int matID, RwInt32 texturenameCase)
{
    Mtl *subMaterial = GetSubMaterial(maxMaterial, matID);

    if (subMaterial)
    {
        RwTexture *rwTex = RpMaterialGetTexture(material);
        if (rwTex && !RwTextureGetMaskName(rwTex))
        {
            Texmap *maskMap = GetStdMatTexmap(maxMaterial, ID_OP);        
            if (maskMap && maskMap->ClassID() == Class_ID(BMTEX_CLASS_ID, 0))                
            {                    
                char *fullMaskName = NULL;            
                char maxMaskName[256];
         
                BitmapTex *maskTexture = (BitmapTex *)maskMap;            
                fullMaskName = maskTexture->GetMapName();
            
                if (fullMaskName)            
                {            
                    char *texName = RwTextureGetName(rwTex);
                    GetFileName(maxMaskName, fullMaskName, texturenameCase);
                    
                    if (strlen(maxMaskName) > (rwTEXTUREBASENAMELENGTH-1))
                    {
                        maxMaskName[rwTEXTUREBASENAMELENGTH-1] = 0;
                    }            
                
                    if (stricmp(texName, maxMaskName) != 0)
                    {                                       
                        RwTextureSetMaskName(rwTex, maxMaskName);                        
                    }
                }            
            }                        
        }
    }
}

RpMaterial *
CreateWhite(void)
{
    RpMaterial *white;    
    RwRGBA cColour;

    white = RpMaterialCreate();
    if (!white)
    {
        return (NULL);
    }
    cColour.red = 255;
    cColour.green = 255;
    cColour.blue = 255;
    cColour.alpha = 255;

    RpMaterialSetColor(white, &cColour);

    return (white);
}

void LimitTextureNameLength( char *TextureName )
{
    //Limit the texture name length to the length allowed by renderware
    //and generate a warning message if the texture name is truncated.
    if (strlen(TextureName) > (rwTEXTUREBASENAMELENGTH-1))
    {
        char msg[1024];

        sprintf(msg, "Texture mask with name %s is > %d characters and is being truncated to ",
            TextureName, rwTEXTUREBASENAMELENGTH-1 );
        TextureName[rwTEXTUREBASENAMELENGTH-1] = 0;
        strcat(msg, TextureName);

        char temp[256];
        sprintf(temp,"RenderWare %s Exporter",VersionNumberText);
        MessageBox(hGMaxHWnd, msg, temp, MB_OK );
    }
}

RwTextureFilterMode MaxFilterToRwFilter( int MaxFilterType )
{
    switch(MaxFilterType)
    {
        case FILTER_NADA:
            return rwFILTERNEAREST;
            break;
        case FILTER_SAT:
            return rwFILTERLINEAR;
            break;
        case FILTER_PYR:
            return rwFILTERLINEARMIPLINEAR;
            break;
        default:
            //maybe generate a warning?
            return rwFILTERNEAREST;
        break;
    }
}

void SetTextureAddressing(int TextureTiling, RwTexture *rwtexture)
{
    RwTextureSetAddressingU(rwtexture, rwTEXTUREADDRESSCLAMP);
    RwTextureSetAddressingV(rwtexture, rwTEXTUREADDRESSCLAMP);
    if (TextureTiling & U_WRAP)
    {
        RwTextureSetAddressingU(rwtexture, rwTEXTUREADDRESSWRAP);
    }
    if (TextureTiling & V_WRAP)
    {
        RwTextureSetAddressingV(rwtexture, rwTEXTUREADDRESSWRAP);
    }
    if (TextureTiling & U_MIRROR)
    {
        RwTextureSetAddressingU(rwtexture, rwTEXTUREADDRESSMIRROR);
    }
    if (TextureTiling & V_MIRROR)
    {
        RwTextureSetAddressingV(rwtexture, rwTEXTUREADDRESSMIRROR);
    }
}

RpMaterial *
Max2RpMaterial(Mtl *maxMaterial, int matID, RwInt32 texturenameCase)
{
    Mtl *subMaterial = NULL;
    RpMaterial *material;
    RwTextureFilterMode filterMode = rwFILTERNEAREST;
    char textureName[256];
    char *fullPathName;

    subMaterial = GetSubMaterial(maxMaterial, matID);
    if (!subMaterial)
    {
        /* Can't convert this return a default white material */        
        return (CreateWhite());
    }

    material = RpMaterialCreate();
    if (material)
    {
        Texmap *maxDiffuse = GetStdMatTexmap(subMaterial, ID_DI);
        Texmap *maxOpacity = GetStdMatTexmap(subMaterial, ID_OP);
        Texmap *maxBump = GetStdMatTexmap(subMaterial,ID_BU);
        RwReal BumpAmount=GetMtlTexmapAmt(subMaterial, ID_BU, 0);
        RwReal ReflAmount=GetMtlTexmapAmt(subMaterial, ID_RL, 0);
        Texmap *maxReflection = GetStdMatTexmap(subMaterial,ID_RL);

        /* colour */
        {
            RwRGBA cColour;

            MaterialGetColour(subMaterial, &cColour, SupportedTexmapClass(maxDiffuse));

            RpMaterialSetColor(material, &cColour);

            {
                RwSurfaceProperties sProps;

                GetRwSurfaceProperties(subMaterial, &sProps);

                RpMaterialSetSurfaceProperties(material, &sProps);
            }
        }

        /* texture */
        {            
            RwTexture *texture;
            RwRaster *raster;
            RwInt32 RasterFlags;
            bool isDiffuseBitmap, isOpacityBitmap, isBumpBitmap, isReflectionBitmap;
            
            isDiffuseBitmap=(maxDiffuse&&maxDiffuse->ClassID()==Class_ID(BMTEX_CLASS_ID,0));
            isOpacityBitmap=(maxOpacity&&maxOpacity->ClassID()==Class_ID(BMTEX_CLASS_ID,0));
            isBumpBitmap=(maxBump&&maxBump->ClassID()==Class_ID(BMTEX_CLASS_ID,0));
            isReflectionBitmap=(maxReflection&&maxReflection->ClassID()==Class_ID(BMTEX_CLASS_ID,0));
            if ((isDiffuseBitmap)&&(isOpacityBitmap||isBumpBitmap))
            {
                RasterFlags=rwRASTERFORMAT8888;
            }
            else
            {
                RasterFlags=rwRASTERFORMAT888;
            }
            if ((isDiffuseBitmap)&&((BitmapTex*)maxDiffuse)->GetFilterType() == FILTER_PYR)
            {
                RasterFlags|= rwRASTERFORMATMIPMAP | rwRASTERFORMATAUTOMIPMAP;
            }
            raster = RwRasterCreate(0, 0, 0, RasterFlags);

            
            texture = RwTextureCreate(raster);
            if (texture)
            {
                //Diffuse is a bitmap
                if (isDiffuseBitmap)
                {
                    filterMode=MaxFilterToRwFilter(
                        ((BitmapTex*)maxDiffuse)->GetFilterType() );
                    RwTextureSetFilterMode(texture, filterMode);
                    SetTextureAddressing(maxDiffuse->GetTextureTiling(), texture);
                    
                    fullPathName = ((BitmapTex*)maxDiffuse)->GetMapName();
                    GetFileName(textureName, fullPathName, texturenameCase);
                    LimitTextureNameLength(textureName);
                    RwTextureSetName(texture, textureName);                    

                    //Create the opacity map in the diffuse's alpha
                    if (isOpacityBitmap)
                    {
                        char maskName[256];

                        if (isOpacityBitmap) {
                            fullPathName = ((BitmapTex*)maxOpacity)->GetMapName();
                        }
                        else {
                            fullPathName = ((BitmapTex*)maxBump)->GetMapName();
                        }
                        GetFileName(maskName, fullPathName, texturenameCase);
                        LimitTextureNameLength(maskName);

                        if (stricmp(textureName, maskName) != 0)
                        {
                            RwTextureSetMaskName(texture, maskName);
                        }
                    }
                    RpMaterialSetTexture(material, texture);
                }
                RwTextureDestroy(texture);
            }

            //Do the bump mapping
            if (isBumpBitmap) {
                RwTexture *BumpTexture;
                RwRaster *BumpRaster;

                if (((BitmapTex*)maxBump)->GetFilterType() == FILTER_PYR)
                {
                    BumpRaster = RwRasterCreate( 0, 0, 0,
                        rwRASTERFORMAT888 | rwRASTERFORMATMIPMAP
                        | rwRASTERFORMATAUTOMIPMAP );                    
                }
                else
                {
                    BumpRaster = RwRasterCreate( 0, 0, 0, rwRASTERFORMAT888 );
                }

                BumpTexture = RwTextureCreate(BumpRaster);
                if (BumpTexture)
                {
                    filterMode=MaxFilterToRwFilter(
                        ((BitmapTex*)maxBump)->GetFilterType() );
                    RwTextureSetFilterMode(BumpTexture, filterMode);
                    SetTextureAddressing(maxBump->GetTextureTiling(), BumpTexture);
                    fullPathName = ((BitmapTex*)maxBump)->GetMapName();
                    GetFileName(textureName, fullPathName, texturenameCase);
                    LimitTextureNameLength(textureName);
                    RwTextureSetName(BumpTexture, textureName);


                    RpMatFXMaterialSetEffects(material, rpMATFXEFFECTBUMPMAP );

                    RpMatFXMaterialSetBumpMapTexture( material, BumpTexture );
    
                    RpMatFXMaterialSetBumpMapCoefficient( material, BumpAmount );

                    RwTextureDestroy(BumpTexture);
                }
            }

            //and now do the environment mapping
            if (isReflectionBitmap) {
                if (((BitmapTex*)maxReflection)->GetFilterType() == FILTER_PYR)
                {
                    raster = RwRasterCreate( 0, 0, 0,
                        rwRASTERFORMAT888 | rwRASTERFORMATMIPMAP
                        | rwRASTERFORMATAUTOMIPMAP );
                }
                else
                {
                    raster = RwRasterCreate( 0, 0, 0, rwRASTERFORMAT888 );
                }

                texture = RwTextureCreate(raster);
                if (texture)
                {
                    filterMode=MaxFilterToRwFilter(
                        ((BitmapTex*)maxReflection)->GetFilterType() );
                    RwTextureSetFilterMode(texture, filterMode);
                    SetTextureAddressing(maxReflection->GetTextureTiling(), texture);
                    fullPathName = ((BitmapTex*)maxReflection)->GetMapName();
                    GetFileName(textureName, fullPathName, texturenameCase);
                    LimitTextureNameLength(textureName);
                    RwTextureSetName(texture, textureName);
                    
                    material=RpMatFXMaterialSetEffects(material,
                        (RpMatFXMaterialFlags)(rpMATFXEFFECTENVMAP | RpMatFXMaterialGetEffects(material)) );
                    material=RpMatFXMaterialSetEnvMapTexture(material,texture);
                    material=RpMatFXMaterialSetEnvMapCoefficient(material,ReflAmount);
                    RwTextureDestroy(texture);
                }
            }
        }
    }

    return (material);
}

/* Converts an absolute pathname to a texture name, i.e
   strip off path and extension. Also allows case of the 
   name to be changed 0:ignore, 1:lower case, 2:upper case */
void
GetFileName(char *outFileName, char *fullPathName, RwInt32 nameCase)
{
    int strLen;
    char *start, *end;
    BOOL strippedext = FALSE;

    if (start = strrchr(fullPathName, '\\'))
    {
        start++;
    }
    else
    {
        start = fullPathName;
    }

    if (end = strrchr(start, '.'))
    {
        *end = '\0';
        strippedext = TRUE;
    }
    else
    {
        end = start + strlen(start);
    }

    strLen = end - start;
    
    strncpy(outFileName, start, strLen + 1);
    outFileName[strLen] = '\0';
    if (strippedext) *end = '.';

    if (nameCase == 1)
    {
        strlwr(outFileName);
    }
    if (nameCase == 2)
    {
        strupr(outFileName);
    }
}
